<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Dispatch.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Dispatch
 * @description     : Manage dispatch.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Dispatch extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Dispatch_Model', 'dispatch', true);
    }

    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Dispatch List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function index() {

        check_permission(VIEW);
        
        $search = $this->input->get('search');
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($search) {
            $config['base_url'] = site_url('frontoffice/dispatch/?search=') . $search;
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->dispatch->get_dispatch_list($config['per_page'], $offset, $search, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['dispatchs'] = $this->dispatch->get_dispatch_list($config['per_page'], $offset, $search, $count = FALSE);

        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['search'] = $search;

        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('dispatch') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('dispatch/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add new Dispatch" user interface
     *                    and process to store "dispatchs" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add() {

        check_permission(ADD);
        if ($_POST) {
            $this->_prepare_dispatch_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_dispatch_data();
                $insert_id = $this->dispatch->insert('dispatchs', $data);
                if ($insert_id) {
                    create_log('Has been created a Dispatch : ' . $data['name']);
                    success($this->lang->line('insert_success'));
                    redirect('frontoffice/dispatch');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('frontoffice/dispatch/add');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('dispatch') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('dispatch/add', $this->data);
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "Dispatch" user interface
     *                    with populate "Dispatch" value
     *                    and process to update "dispatchs" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit($id = null) {

        check_permission(EDIT);
        if ($_POST) {
            $this->_prepare_dispatch_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_dispatch_data();
                $updated = $this->dispatch->update('dispatchs', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a Dispatch : ' . $data['name']);
                    success($this->lang->line('update_success'));
                    redirect('frontoffice/dispatch');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('frontoffice/dispatch/edit/' . $this->input->post('id'));
                }
            } else {
                $this->data['dispatch'] = $this->dispatch->get_single('dispatchs', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['dispatch'] = $this->dispatch->get_single('dispatchs', array('id' => $id));
                if (!$this->data['dispatch']) {
                    redirect('frontoffice/dispatch');
                }
            }
        }

        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('dispatch') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('dispatch/edit', $this->data);
    }

    /*****************Function _prepare_dispatch_validation**********************************
     * @type            : Function
     * @function name   : _prepare_dispatch_validation
     * @description     : Process "Dispatch" user input data validation
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    private function _prepare_dispatch_validation() {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
        $this->form_validation->set_rules('to_title', $this->lang->line('to_title'), 'trim|required');
        $this->form_validation->set_rules('date', $this->lang->line('date'), 'trim|required');
    }

    /*****************Function _get_posted_dispatch_data**********************************
     * @type            : Function
     * @function name   : _get_posted_dispatch_data
     * @description     : Prepare "Dispatch" user input data to save into database
     *
     * @param           : null
     * @return          : $data array(); value
     * ********************************************************** */
    private function _get_posted_dispatch_data() {

        $items = array();
        $items[] = 'to_title';
        $items[] = 'reference';
        $items[] = 'from_title';
        $items[] = 'address';
        $items[] = 'note';
        $data = elements($items, $_POST);

        $data['date'] = $this->input->post('date') ? date('Y-m-d', strtotime($this->input->post('date'))) : date('Y-m-d');

        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {
            $data['status'] = 1;
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();
        }

        if ($_FILES['attachment']['name']) {
            $data['attachment'] = $this->_upload_attachment();
        }

        return $data;
    }

    /*     * ***************Function _upload_attachment**********************************
     * @type            : Function
     * @function name   : _upload_attachment
     * @description     : Process to upload dispatch attachment into server
     *                     and return attachment file name
     * @param           : null
     * @return          : $return_attachment string value
     * ********************************************************** */

    private function _upload_attachment() {

        $prev_attachment = $this->input->post('prev_attachment');
        $attachment = $_FILES['attachment']['name'];
        $attachment_type = $_FILES['attachment']['type'];
        $return_attachment = '';

        if ($attachment != "") {
            if (
                $attachment_type == 'application/vnd.openxmlformats-officedocument.wordprocessingml.document' ||
                $attachment_type == 'application/msword' || $attachment_type == 'text/plain' ||
                $attachment_type == 'application/vnd.ms-office' || $attachment_type == 'application/pdf' ||
                $attachment_type == 'image/jpeg' || $attachment_type == 'image/pjpeg' ||
                $attachment_type == 'image/jpg' || $attachment_type == 'image/png' ||
                $attachment_type == 'image/x-png' || $attachment_type == 'image/gif'
            ) {

                $destination = 'assets/uploads/attachment/';

                $file_type = explode(".", $attachment);
                $extension = strtolower($file_type[count($file_type) - 1]);
                $attachment_path = 'dispatch-attachment-' . time() . '-sbiz.' . $extension;

                move_uploaded_file($_FILES['attachment']['tmp_name'], $destination . $attachment_path);

                // need to unlink previous photo
                if ($prev_attachment != "") {
                    if (file_exists($destination . $prev_attachment)) {
                        @unlink($destination . $prev_attachment);
                    }
                }

                $return_attachment = $attachment_path;
            }
        } else {
            $return_attachment = $prev_attachment;
        }

        return $return_attachment;
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Dispatch" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('frontoffice/dispatch');
        }

        $dispatch = $this->dispatch->get_single('dispatchs', array('id' => $id));

        if ($this->dispatch->delete('dispatchs', array('id' => $id))) {
            // delete attachment
            $destination = 'assets/uploads/';
            if (file_exists($destination . '/attachment/' . $dispatch->attachment)) {
                @unlink($destination . '/attachment/' . $dispatch->attachment);
            }
            create_log('Has been deleted a Dispatch : ' . $dispatch->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect('frontoffice/dispatch');
    }
}
