<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Charge.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Charge
 * @description     : Manage charge.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Charge extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Charge_Model', 'charge', true);
    }

    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Charge List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function index() {

        check_permission(VIEW);

        $search = trim($this->input->get('search'));

        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($search) {
            $config['base_url'] = site_url('doctor/charge/?search=') . $search;
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->charge->get_charge_list($config['per_page'], $offset, $search, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['charges'] = $this->charge->get_charge_list($config['per_page'], $offset, $search, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['search'] = $search;

        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_charge') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('charge/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add new Charge" user interface
     *                    and process to store "charges" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add() {

        check_permission(ADD);
        if ($_POST) {
            $this->_prepare_charge_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_charge_data();
                $insert_id = $this->charge->insert('doctor_charges', $data);
                if ($insert_id) {
                    create_log('Has been created a Charge : ' . $data['first_fee']);
                    success($this->lang->line('insert_success'));
                    redirect('doctor/charge');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('doctor/charge/add');
                }
            } else {
                $this->data = $_POST;
            }
        }
        $this->data['doctors'] = $this->charge->get_doctor_list();
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('charge') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('charge/add', $this->data);
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "Charge" user interface
     *                    with populate "Charge" value
     *                    and process to update "charges" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit($id = null) {

        check_permission(EDIT);
        if ($_POST) {
            $this->_prepare_charge_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_charge_data();
                $updated = $this->charge->update('doctor_charges', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a Charge : ' . $data['first_fee']);
                    success($this->lang->line('update_success'));
                    redirect('doctor/charge');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('doctor/charge/edit/' . $this->input->post('id'));
                }
            } else {
                $this->data['charge'] = $this->charge->get_single('doctor_charges', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['charge'] = $this->charge->get_single('doctor_charges', array('id' => $id));
                if (!$this->data['charge']) {
                    redirect('doctor/charge');
                }
            }
        }
        $this->data['doctors'] = $this->charge->get_doctor_list();
        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('charge') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('charge/edit', $this->data);
    }

    /*****************Function _prepare_charge_validation**********************************
     * @type            : Function
     * @function name   : _prepare_charge_validation
     * @description     : Process "Charge" user input data validation
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    private function _prepare_charge_validation() {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');

        $this->form_validation->set_rules('doctor_id', $this->lang->line('doctor') . ' ' . $this->lang->line('name'), 'trim|required|callback_doctor_name');
        $this->form_validation->set_rules('first_fee', $this->lang->line('first_fee'), 'trim|required');
        $this->form_validation->set_rules('second_fee', $this->lang->line('second_fee'), 'trim|required');
        $this->form_validation->set_rules('hospital_commission', $this->lang->line('hospital') . ' ' . $this->lang->line('commission'), 'trim');
        $this->form_validation->set_rules('first_fee_day', $this->lang->line('first_fee_day'), 'trim|required');
    }

    /*****************Function category_name**********************************
     * @type            : Function
     * @function name   : category_name
     * @description     : Unique check for "Charge Category Name" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function doctor_name() {
        if ($this->input->post('id') == '') {
            $doctor_name = $this->charge->duplicate_check($this->input->post('doctor_id'));
            if ($doctor_name) {
                $this->form_validation->set_message('doctor_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $doctor_name = $this->charge->duplicate_check($this->input->post('doctor_id'), $this->input->post('id'));
            if ($doctor_name) {
                $this->form_validation->set_message('doctor_name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function _get_posted_charge_data**********************************
     * @type            : Function
     * @function name   : _get_posted_charge_data
     * @description     : Prepare "Charge" user input data to save into database
     *
     * @param           : null
     * @return          : $data array(); value
     * ********************************************************** */
    private function _get_posted_charge_data() {

        $items = array();
        $items[] = 'doctor_id';
        $items[] = 'ipd_charge';
        $items[] = 'ot_charge';
        $items[] = 'first_fee';
        $items[] = 'first_fee_day';
        $items[] = 'second_fee';
        $items[] = 'hospital_commission';
        $items[] = 'serial_no';
        $items[] = 'note';
        $items[] = 'sort_by';
        $data = elements($items, $_POST);
        
        $data['is_opd'] = $this->input->post('is_opd') ? $this->input->post('is_opd') : 0;
        $data['status'] = $this->input->post('status') ? $this->input->post('status') : 0;

        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();
        }
        return $data;
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Charge" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('doctor/charge');
        }

        $charge = $this->charge->get_single('doctor_charges', array('id' => $id));

        if ($this->charge->delete('doctor_charges', array('id' => $id))) {
            create_log('Has been deleted a Charge : ' . $charge->first_fee);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }
}
