<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Frontend.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Frontend
 * @description     : Manage frontend information of the hospital.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Frontend extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Frontend_Model', 'frontend', true);
    }

    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Frontend List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function index() {

        check_permission(VIEW);

        $search = $this->input->get('search');
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($search) {
            $config['base_url'] = site_url('frontend/?search=') . $search;
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->frontend->get_page_list($config['per_page'], $offset, $search, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['pages'] = $this->frontend->get_page_list($config['per_page'], $offset, $search, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['search'] = $search;

        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_frontend') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('frontend/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add new Page" user interface
     *                    and process to store "pages" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add() {

        check_permission(ADD);

        if ($_POST) {
            $this->_prepare_frontend_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_frontend_data();
                $insert_id = $this->frontend->insert('pages', $data);
                if ($insert_id) {
                    create_log('Has been created a frontend page : ' . $data['title']);
                    success($this->lang->line('insert_success'));
                    redirect('frontend');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('frontend/add');
                }
            } else {
                $this->data = $_POST;
            }
        }
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('page') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('frontend/add', $this->data);
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "Frontend Page" user interface
     *                    with populate "Frontend Page" value
     *                    and process to update "pages" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit($id = null) {

        check_permission(EDIT);

        if ($_POST) {
            $this->_prepare_frontend_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_frontend_data();
                $updated = $this->frontend->update('pages', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a frontend page : ' . $data['title']);
                    success($this->lang->line('update_success'));
                    redirect('frontend');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('frontend/edit/' . $this->input->post('id'));
                }
            } else {
                $this->data['page'] = $this->frontend->get_single('pages', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['page'] = $this->frontend->get_single('pages', array('id' => $id));
                if (!$this->data['page']) {
                    redirect('frontend');
                }
            }
        }
        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('page') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('frontend/edit', $this->data);
    }

    /*****************Function _prepare_frontend_validation**********************************
     * @type            : Function
     * @function name   : _prepare_frontend_validation
     * @description     : Process "pages" user input data validation
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    private function _prepare_frontend_validation() {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
        $this->form_validation->set_rules('title', $this->lang->line('title'), 'trim|required');
    }

    /*****************Function _get_posted_frontend_data**********************************
     * @type            : Function
     * @function name   : _get_posted_frontend_data
     * @description     : Prepare "pages" user input data to save into database
     *
     * @param           : null
     * @return          : $data array(); value
     * ********************************************************** */
    private function _get_posted_frontend_data() {

        $items = array();
        $items[] = 'title';
        $items[] = 'bn_title';
        $items[] = 'description';
        $items[] = 'bn_description';
        $items[] = 'meta_title';
        $items[] = 'meta_keyword';
        $items[] = 'meta_description';
        $data = elements($items, $_POST);

        $data['sidebar'] = $this->input->post('sidebar') ?? 0;
        $data['image_view'] = $this->input->post('image_view') ?? 0;
        $data['image_full_width'] = $this->input->post('image_full_width') ?? 0;
        $data['is_view_pdf'] = $this->input->post('is_view_pdf') ?? 0;


        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {
            $data['status'] = 1;
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();
        }

        if (isset($_FILES['image']['name'])) {
            $data['image'] = $this->_upload_image();
        }

        if (isset($_FILES['attachment']['name'])) {
            $data['attachment'] = $this->_upload_attachment();
        }

        return $data;
    }


    /*     * ***************Function _upload_image**********************************
     * @type            : Function
     * @function name   : _upload_image
     * @description     : Process to to upload frontend image in the server
     *                    and return image name                   
     *                       
     * @param           : null
     * @return          : $return_image string value 
     * ********************************************************** */

    private function _upload_image() {

        $prev_image = $this->input->post('prev_image');
        $image = $_FILES['image']['name'];
        $image_type = $_FILES['image']['type'];
        $return_image = '';
        if ($image != "") {
            if (
                $image_type == 'image/jpeg' || $image_type == 'image/pjpeg' ||
                $image_type == 'image/jpg' || $image_type == 'image/png' ||
                $image_type == 'image/x-png' || $image_type == 'image/gif'
            ) {

                $destination = 'assets/uploads/page/';

                $file_type = explode(".", $image);
                $extension = strtolower($file_type[count($file_type) - 1]);
                $image_path = 'page-image-' . time() . '-hms.' . $extension;

                move_uploaded_file($_FILES['image']['tmp_name'], $destination . $image_path);

                // need to unlink previous image
                if ($prev_image != "") {
                    if (file_exists($destination . $prev_image)) {
                        @unlink($destination . $prev_image);
                    }
                }

                $return_image = $image_path;
            }
        } else {
            $return_image = $prev_image;
        }

        return $return_image;
    }

    /*     * ***************Function _upload_attachment**********************************
     * @type            : Function
     * @function name   : _upload_attachment
     * @description     : Process to to upload frontend attachment in the server
     *                    and return attachment name                   
     *                       
     * @param           : null
     * @return          : $return_attachment string value 
     * ********************************************************** */

    private function _upload_attachment() {

        $prev_attachment = $this->input->post('prev_attachment');
        $attachment = $_FILES['attachment']['name'];
        $attachment_type = $_FILES['attachment']['type'];
        $return_attachment = '';
        if ($attachment != "") {
            if (
                $attachment_type == 'image/jpeg' || $attachment_type == 'image/pjpeg' ||
                $attachment_type == 'image/jpg' || $attachment_type == 'image/png' ||
                $attachment_type == 'image/x-png' || $attachment_type == 'image/gif' ||
                $attachment_type == 'application/vnd.openxmlformats-officedocument.wordprocessingml.document' ||
                $attachment_type == 'application/msword' || $attachment_type == 'text/plain' ||
                $attachment_type == 'application/vnd.ms-office' || $attachment_type == 'application/pdf'
            ) {

                $destination = 'assets/uploads/page/';

                $file_type = explode(".", $attachment);
                $extension = strtolower($file_type[count($file_type) - 1]);
                $attachment_path = 'page-attachment-' . time() . '-hms.' . $extension;

                move_uploaded_file($_FILES['attachment']['tmp_name'], $destination . $attachment_path);

                // need to unlink previous attachment
                if ($prev_attachment != "") {
                    if (file_exists($destination . $prev_attachment)) {
                        @unlink($destination . $prev_attachment);
                    }
                }

                $return_attachment = $attachment_path;
            }
        } else {
            $return_attachment = $prev_attachment;
        }

        return $return_attachment;
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Pages" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('frontend');
        }

        $frontend = $this->frontend->get_single('pages', array('id' => $id));

        if ($this->frontend->delete('pages', array('id' => $id))) {
            // delete image and attachment
            $destination = 'assets/uploads/';
            if (file_exists($destination . '/page/' . $frontend->image)) {
                @unlink($destination . '/page/' . $frontend->image);
            }
            if (file_exists($destination . '/page/' . $frontend->attachment)) {
                @unlink($destination . '/page/' . $frontend->attachment);
            }
            create_log('Has been deleted a frontend page : ' . $frontend->title);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }

    /*****************Function section**********************************
     * @type            : Function
     * @function name   : section
     * @description     : Load "section" user interface
     *                    and process to store "sections" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function section() {

        check_permission(ADD);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('doctor', $this->lang->line('doctor'), 'trim');
            $this->form_validation->set_rules('department', $this->lang->line('department'), 'trim');
            $this->form_validation->set_rules('photo', $this->lang->line('photo'), 'trim');

            if ($this->form_validation->run() === TRUE) {
                $data['news'] = $this->input->post('news');
                $data['bn_news'] = $this->input->post('bn_news');
                $data['doctor'] = $this->input->post('doctor');
                $data['bn_doctor'] = $this->input->post('bn_doctor');
                $data['department'] = $this->input->post('department');
                $data['bn_department'] = $this->input->post('bn_department');
                $data['photo'] = $this->input->post('photo');
                $data['bn_photo'] = $this->input->post('bn_photo');
                $data['video'] = $this->input->post('video');
                $data['bn_video'] = $this->input->post('bn_video');
                $data['status'] = 1;
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();

                $section = $this->frontend->get_single('sections', array());
                if (isset($section) && !empty($section)) {
                    $this->frontend->update('sections', $data, array('id' => $section->id));
                    create_log('Has been update a frontend section');
                    success($this->lang->line('update_success'));
                    redirect('frontend/section');
                } else {
                    $this->frontend->insert('sections', $data);
                    create_log('Has been created a frontend section');
                    success($this->lang->line('insert_success'));
                    redirect('frontend/section');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['section'] = $this->frontend->get_single('sections', array());
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('section') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('frontend/section', $this->data);
    }


    /*****************Function about**********************************
     * @type            : Function
     * @function name   : about
     * @description     : Load "about" user interface
     *                    and process to store "sections" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function about() {

        check_permission(ADD);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('about', $this->lang->line('about'), 'trim');

            if ($this->form_validation->run() === TRUE) {
                $data['about'] = $this->input->post('about');
                $data['bn_about'] = $this->input->post('bn_about');
                $data['appointment'] = $this->input->post('appointment');
                $data['bn_appointment'] = $this->input->post('bn_appointment');
                $data['why_choose_us'] = $this->input->post('why_choose_us');
                $data['bn_why_choose_us'] = $this->input->post('bn_why_choose_us');
                $data['service'] = $this->input->post('service');
                $data['bn_service'] = $this->input->post('bn_service');
                $data['testimonial'] = $this->input->post('testimonial');
                $data['bn_testimonial'] = $this->input->post('bn_testimonial');
                $data['contact'] = $this->input->post('contact');
                $data['bn_contact'] = $this->input->post('bn_contact');
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();

                $about = $this->frontend->get_single('sections', array());
                if (isset($about) && !empty($about)) {
                    $this->frontend->update('sections', $data, array('id' => $about->id));
                    create_log('Has been update a frontend section');
                    success($this->lang->line('update_success'));
                    redirect('frontend/about');
                } else {
                    $this->frontend->insert('sections', $data);
                    create_log('Has been created a frontend section');
                    success($this->lang->line('insert_success'));
                    redirect('frontend/about');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['about'] = $this->frontend->get_single('sections', array());
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('section') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('frontend/about', $this->data);
    }

    /*****************Function support**********************************
     * @type            : Function
     * @function name   : section
     * @description     : Load "support" user interface
     *                    and process to store "sections" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function support() {

        check_permission(ADD);

        if ($_POST) {

            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('support', $this->lang->line('support'), 'trim');

            if ($this->form_validation->run() === TRUE) {
                $data['support'] = $this->input->post('support');
                $data['bn_support'] = $this->input->post('bn_support');
                $data['expert_doctor'] = $this->input->post('expert_doctor');
                $data['bn_expert_doctor'] = $this->input->post('bn_expert_doctor');
                $data['online_payment'] = $this->input->post('online_payment');
                $data['bn_online_payment'] = $this->input->post('bn_online_payment');
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();

                $support = $this->frontend->get_single('sections', array());
                if (isset($support) && !empty($support)) {
                    $this->frontend->update('sections', $data, array('id' => $support->id));
                    create_log('Has been update a frontend section');
                    success($this->lang->line('update_success'));
                    redirect('frontend/support');
                } else {
                    $this->frontend->insert('sections', $data);
                    create_log('Has been created a frontend section');
                    success($this->lang->line('insert_success'));
                    redirect('frontend/support');
                }
            } else {
                $this->data = $_POST;
            }
        }

        $this->data['support'] = $this->frontend->get_single('sections', array());
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('support') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('frontend/support', $this->data);
    }
}
