<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Designation.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Designation
 * @description     : Manage designation.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Designation extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Designation_Model', 'designation', true);
    }

    /*****************Function index**********************************
     * @type            : Function
     * @function name   : index
     * @description     : Load "Designation List" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function index() {

        check_permission(VIEW);

        $search = $this->input->get('search');
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($search) {
            $config['base_url'] = site_url('hrm/designation/?search=') . $search;
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->designation->get_designation_list($config['per_page'], $offset, $search, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['designations'] = $this->designation->get_designation_list($config['per_page'], $offset, $search, $count = FALSE);

        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['search'] = $search;

        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('manage_designation') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('designation/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add new Designation" user interface
     *                    and process to store "designations" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function add() {

        check_permission(ADD);
        if ($_POST) {
            $this->_prepare_designation_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_designation_data();
                $insert_id = $this->designation->insert('designations', $data);
                if ($insert_id) {
                    create_log('Has been created a Designation : ' . $data['name']);
                    success($this->lang->line('insert_success'));
                    redirect('hrm/designation');
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('hrm/designation/add');
                }
            } else {
                $this->data = $_POST;
            }
        }
        $this->data['departments'] = $this->designation->get_list('departments', array('status' => 1), '', '', '', 'sort_by', 'ASC');
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add') . ' ' . $this->lang->line('designation') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('designation/add', $this->data);
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "Designation" user interface
     *                    with populate "Designation" value
     *                    and process to update "designations" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit($id = null) {

        check_permission(EDIT);
        if ($_POST) {
            $this->_prepare_designation_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_designation_data();
                $updated = $this->designation->update('designations', $data, array('id' => $this->input->post('id')));
                if ($updated) {
                    create_log('Has been updated a Designation : ' . $data['name']);
                    success($this->lang->line('update_success'));
                    redirect('hrm/designation');
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('hrm/designation/edit/' . $this->input->post('id'));
                }
            } else {
                $this->data['designation'] = $this->designation->get_single('designations', array('id' => $this->input->post('id')));
            }
        } else {
            if ($id) {
                $this->data['designation'] = $this->designation->get_single('designations', array('id' => $id));
                if (!$this->data['designation']) {
                    redirect('hrm/designation');
                }
            }
        }
        $this->data['departments'] = $this->designation->get_list('departments', array('status' => 1), '', '', '', 'sort_by', 'ASC');
        $this->data['edit'] = TRUE;
        $this->layout->title($this->lang->line('edit') . ' ' . $this->lang->line('designation') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('designation/edit', $this->data);
    }

    /*****************Function _prepare_designation_validation**********************************
     * @type            : Function
     * @function name   : _prepare_designation_validation
     * @description     : Process "Designation" user input data validation
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    private function _prepare_designation_validation() {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
        $this->form_validation->set_rules('department_id', $this->lang->line('department') . ' ' . $this->lang->line('name'), 'trim|required');
        $this->form_validation->set_rules('name', $this->lang->line('designation') . ' ' . $this->lang->line('name'), 'trim|required|callback_name');
        $this->form_validation->set_rules('bn_name', $this->lang->line('designation') . ' ' . $this->lang->line('bn_name'), 'trim');
    }

    /*****************Function name**********************************
     * @type            : Function
     * @function name   : name
     * @description     : Unique check for "Designation Name" data/value
     *
     * @param           : null
     * @return          : boolean true/false
     * ********************************************************** */
    public function name() {
        if ($this->input->post('id') == '') {
            $designation = $this->designation->duplicate_check($this->input->post('name'));
            if ($designation) {
                $this->form_validation->set_message('name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else if ($this->input->post('id') != '') {
            $designation = $this->designation->duplicate_check($this->input->post('name'), $this->input->post('id'));
            if ($designation) {
                $this->form_validation->set_message('name', $this->lang->line('already_exist'));
                return FALSE;
            } else {
                return TRUE;
            }
        } else {
            return TRUE;
        }
    }

    /*****************Function _get_posted_designation_data**********************************
     * @type            : Function
     * @function name   : _get_posted_designation_data
     * @description     : Prepare "Designation" user input data to save into database
     *
     * @param           : null
     * @return          : $data array(); value
     * ********************************************************** */
    private function _get_posted_designation_data() {

        $items = array();
        $items[] = 'department_id';
        $items[] = 'name';
        $items[] = 'bn_name';
        $items[] = 'note';
        $items[] = 'sort_by';
        $data = elements($items, $_POST);

        $data['status'] = $this->input->post('status') ?? 0;
        $data['is_view_on_web'] = $this->input->post('is_view_on_web') ?? 0;

        if ($this->input->post('id')) {
            $data['modified_at'] = date('Y-m-d H:i:s');
            $data['modified_by'] = logged_in_user_id();
        } else {
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['created_by'] = logged_in_user_id();
        }
        return $data;
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Designation" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect('hrm/designation');
        }

        $designation = $this->designation->get_single('designations', array('id' => $id));

        if ($this->designation->delete('designations', array('id' => $id))) {
            create_log('Has been deleted a Designation : ' . $designation->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect('hrm/designation');
    }
}
