<?php

defined('BASEPATH') or exit('No direct script access allowed');

/* * *****************Surgeon.php**********************************
 * @product name    : Hospital Management System
 * @type            : Class
 * @class name      : Surgeon
 * @description     : Manage IPD Surgeon Chart.
 * @author          : South Bangla IT Zone Team
 * @url             : http://hms.sbitzone.com
 * @support         : rabibd.it@gmail.com
 * @copyright       : South Bangla IT Zone Team
 * ********************************************************** */

class Surgeon extends MY_Controller {

    public $data = array();

    function __construct() {
        parent::__construct();
        $this->load->model('Surgeon_Model', 'surgeon', true);
        $this->load->model('Ipd_Model', 'ipd', true);
        $this->data['doctors'] = $this->ipd->get_doctor_list();
    }

    public function index($ipd_id) {

        check_permission(VIEW);

        $this->data['patient'] = $this->ipd->get_single_ipd_patient($ipd_id);
        $this->data['total_investigation_due'] = $this->ipd->get_total_investigation_due($this->data['patient']->patient_id);
        $this->data['total_pharmacy_due'] = $this->ipd->get_total_pharmacy_due($this->data['patient']->patient_id);
        $this->data['total_ipd_advanced'] = $this->ipd->get_total_ipd_advaced($ipd_id, $this->data['patient']->patient_id);
        

        $search = trim($this->input->get('search'));
        $config = array();
        $config = get_pagination(ADMIN_PER_PAGE);
        if ($search) {
            $config['base_url'] = site_url('ipd/surgeon/index/' . $ipd_id . '/?search=' . $search);
        } else {
            $config['base_url'] = current_url();
        }
        if ($this->input->get('page')) {
            $sgm = (int) trim($this->input->get('page'));
            $offset = $config['per_page'] * ($sgm - 1);
        } else {
            $offset = 0;
        }

        $config['total_rows'] = $this->surgeon->get_patient_surgeon_list($config['per_page'], $offset, $ipd_id, $search, $count = TRUE);
        $this->pagination->initialize($config);
        $this->data['surgeons'] = $this->surgeon->get_patient_surgeon_list($config['per_page'], $offset, $ipd_id, $search, $count = FALSE);
        $this->data['sl_count'] = ($this->input->get('page') == 0 ? 1 : (($this->input->get('page') - 1) * $config["per_page"] + 1));
        $this->data['search'] = $search;

        $this->data['list'] = TRUE;
        $this->layout->title($this->lang->line('surgeon') . ' ' . $this->lang->line('chart') . ' | ' . $this->global_setting->brand_title);
        $this->layout->view('surgeon/index', $this->data);
    }

    /*****************Function add**********************************
     * @type            : Function
     * @function name   : add
     * @description     : Load "Add Surgeon" user interface
     *                    and process to store "ipd_surgeon" into database
     * @param           : null
     * @return          : null
     * ********************************************************** */

    public function add() {

        check_permission(ADD);

        if ($_POST) {
            $this->load->library('form_validation');
            $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
            $this->form_validation->set_rules('doctor_id', $this->lang->line('surgeon') . ' ' . $this->lang->line('name'), 'trim|required');
            $this->form_validation->set_rules('charge', $this->lang->line('doctor') . ' ' . $this->lang->line('charge'), 'trim|required');
            if ($this->form_validation->run() === TRUE) {
                $data['ipd_id'] = $this->input->post('ipd_id');
                $data['patient_id'] = $this->input->post('patient_id');
                $data['type'] = 'surgeon';
                $data['doctor_id'] = $this->input->post('doctor_id');
                $data['charge'] = $this->input->post('charge');
                $data['date'] = date('Y-m-d H:i', strtotime($this->input->post('date')));
                $data['instruction'] = $this->input->post('instruction');
                $data['surgeon_status'] = 'complete';
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['created_by'] = logged_in_user_id();
                $insert_id = $this->surgeon->insert('ipd_surgeons', $data);
                if ($insert_id) {
                    $patient = $this->surgeon->get_single('patients', array('user_id' => $data['patient_id']));
                    create_log('Has been created a surgeon : ' . $patient->name);
                    success($this->lang->line('insert_success'));
                    redirect('ipd/surgeon/index/' . $data['ipd_id']);
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('ipd/surgeon/index/' . $data['ipd_id']);
                }
            } else {
                $this->data = $_POST;
            }
        }
    }

    /*****************Function get_surgeon**********************************
     * @type            : Function
     * @function name   : get_surgeon
     * @description     : Load "Single Surgeon" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function get_surgeon($id) {
        if ($id > 0) {
            $surgeon = $this->surgeon->get_single('ipd_surgeons', array('id' => $id));
            echo (json_encode($surgeon));
        }
    }

    /*****************Function edit**********************************
     * @type            : Function
     * @function name   : edit
     * @description     : Load Update "Surgeon Chart" user interface
     *                    with populate "Surgeon Chart" value
     *                    and process to update "ipd_surgeons" into database
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function edit() {

        check_permission(EDIT);

        if ($_POST) {
            $this->load->library('form_validation');
            $this->form_validation->set_rules('doctor_id', $this->lang->line('surgeon') . ' ' . $this->lang->line('name'), 'trim|required');
            $this->form_validation->set_rules('charge', $this->lang->line('doctor') . ' ' . $this->lang->line('charge'), 'trim|required');
            if ($this->form_validation->run() === TRUE) {
                $data['type'] = 'surgeon';
                $data['doctor_id'] = $this->input->post('doctor_id');
                $data['charge'] = $this->input->post('charge');
                $data['date'] = date('Y-m-d H:i', strtotime($this->input->post('date')));
                $data['instruction'] = $this->input->post('instruction');
                $data['surgeon_status'] = $this->input->post('surgeon_status');
                $data['note'] = $this->input->post('note');
                $data['modified_at'] = date('Y-m-d H:i:s');
                $data['modified_by'] = logged_in_user_id();

                $this->surgeon->update('ipd_surgeons', $data, array('id' => $this->input->post('surgeon_id')));
                $patient = $this->surgeon->get_single('patients', array('user_id' => $this->input->post('patient_id')));
                create_log('Has been updated a surgeon : ' . $patient->name);
                $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('update_success'));
            } else {
                $msg = array(
                    'doctor_id' => form_error('doctor_id'),
                    'date' => form_error('date'),
                );
                $array = array('status' => 'failed', 'error' => $msg, 'message' => '');
            }

            echo json_encode($array);
        }
    }

    /*     * ***************Function get_single_surgeon**********************************
     * @type            : Function
     * @function name   : get_single_surgeon
     * @description     : "Load single surgeon information" from database                  
     *                    to the user interface   
     * @param           : null
     * @return          : null 
     * ********************************************************** */

    public function get_single_surgeon() {
        $id = $this->input->post('id');
        $this->data['surgeon'] = $this->surgeon->get_single_surgeon($id);
        echo $this->load->view('ipd/surgeon/view', $this->data);
    }

    /*****************Function get_doctor_charge**********************************
     * @type            : Function
     * @function name   : get_doctor_charge
     * @description     : Load "IPD Doctor Charge" user interface
     *
     * @param           : null
     * @return          : null
     * ********************************************************** */
    public function get_doctor_charge() {
        $user_id = $this->input->post('user_id');
        if ($user_id > 0) {
            $doctor_charges = $this->surgeon->get_single('doctor_charges', array('doctor_id' => $user_id));
            $doctor_charge = array(
                'charge' => $doctor_charges->ot_charge
            );
        }
        echo (json_encode($doctor_charge));
    }

    /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Surgeon" data from database
     *
     * @param           : $id integer value
     * @return          : null
     * ********************************************************** */
    public function delete($id = null) {

        check_permission(DELETE);

        if (!is_numeric($id)) {
            error($this->lang->line('unexpected_error'));
            redirect($_SERVER['HTTP_REFERER']);
        }

        $surgeon = $this->surgeon->get_single('ipd_surgeons', array('id' => $id));
        $patient = $this->surgeon->get_single('patients', array('user_id' => $surgeon->patient_id));

        if ($this->ipd->delete('ipd_surgeons', array('id' => $id))) {
            create_log('Has been deleted a surgeon : ' . $patient->name);
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect($_SERVER['HTTP_REFERER']);
    }
}
